﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace YumeNikkiRandomizer
{
    class Page : RPGByteData
    {
        public int pageNum = 0;
        public PageConditions conditions; // 02
        string charSet = ""; // 15
        int charIndex = 0; // 16
        int charDir = 2; // 17
        int charPattern = 1; // 18
        bool transparent = false; // 19
        int moveType = 1; // 1f
        int moveFrequency = 3; // 20
        int startTrigger = 0; // 21
        int layer = 0; // 22
        bool forbidOverlap = false; // 23
        public int animationType = 0; // 24
        public int animationFrequency = 3; // 25
        PageCustomRoute customRoute; // 29
        public List<Command> commands; // 33 length, 34 content
        
        static string myClass = "Page";
        Chunks chunks;
        
        public Page(FileStream f)
        {
            load(f);
        }
        public Page(int num)
        {
            pageNum = num;
            chunks = new Chunks();
            
            conditions = new PageConditions();
            commands = new List<Command>();
            commands.Add(new Command(Command.C_TERMINATOR, 0, new int[] { }));
            chunks.add(0x02); // conditions
            chunks.add(0x33); // commands length
            chunks.add(0x34); // commands
        }
        public Page()
        {
            chunks = new Chunks();
            
            conditions = new PageConditions();
            commands = new List<Command>();
            commands.Add(new Command(Command.C_TERMINATOR, 0, new int[] { }));
            chunks.add(0x02); // conditions
            chunks.add(0x33); // commands length
            chunks.add(0x34); // commands
        }
        
        // Loads a single page within a map/troop event.
        override public void load(FileStream f)
        {
            chunks = new Chunks(f, myClass);
            
            pageNum = M.readMultibyte(f);
            
            if (chunks.next(0x02))
            {
                M.readMultibyte(f); // Length
                conditions = new PageConditions(f);
            }
            
            if (chunks.next(0x15))
                charSet = M.readString(f, M.S_FILENAME);
            if (chunks.next(0x16))
                charIndex = M.readLengthMultibyte(f);
            if (chunks.next(0x17))
                charDir = M.readLengthMultibyte(f);
            if (chunks.next(0x18))
                charPattern = M.readLengthMultibyte(f);
            if (chunks.next(0x19))
                transparent = M.readLengthBool(f);
            
            if (chunks.next(0x1f))
                moveType = M.readLengthMultibyte(f);
            if (chunks.next(0x20))
                moveFrequency = M.readLengthMultibyte(f);
            if (chunks.next(0x21))
                startTrigger = M.readLengthMultibyte(f);
            if (chunks.next(0x22))
                layer = M.readLengthMultibyte(f);
            if (chunks.next(0x23))
                forbidOverlap = M.readLengthBool(f);
            if (chunks.next(0x24))
                animationType = M.readLengthMultibyte(f);
            if (chunks.next(0x25))
                animationFrequency = M.readLengthMultibyte(f);
            
            if (chunks.next(0x29))
                customRoute = new PageCustomRoute(f);
            
            if (chunks.next(0x33))
                M.readLengthMultibyte(f); // Command section length
            
            if (chunks.next(0x34))
                commands = M.readCommandList(f);
            
            M.byteCheck(f, 0x00);
        }
        
        // Checks commands for teleports.
        public void getWarps()
        {
            M.currentPageNum = pageNum;
            M.currentPage = pageNum.ToString();
            
            for (int i = 0; i < commands.Count; i++)
            {
                M.currentLine = "Line " + (i + 1);
                commands[i].getWarps();
            }
        }
       
        // Writes page data.
        override protected void myWrite()
        {
            M.writeMultibyte(pageNum);
            
            if (chunks.wasNext(0x02))
            {
                M.writeMultibyte(conditions.getLength());
                conditions.write();
            }
            
            if (chunks.wasNext(0x15))
                M.writeString(charSet, M.S_FILENAME);
            if (chunks.wasNext(0x16))
                M.writeLengthMultibyte(charIndex);
            if (chunks.wasNext(0x17))
                M.writeLengthMultibyte(charDir);
            if (chunks.wasNext(0x18))
                M.writeLengthMultibyte(charPattern);
            if (chunks.wasNext(0x19))
                M.writeLengthBool(transparent);
            
            if (chunks.wasNext(0x1f))
                M.writeLengthMultibyte(moveType);
            if (chunks.wasNext(0x20))
                M.writeLengthMultibyte(moveFrequency);
            if (chunks.wasNext(0x21))
                M.writeLengthMultibyte(startTrigger);
            if (chunks.wasNext(0x22))
                M.writeLengthMultibyte(layer);
            if (chunks.wasNext(0x23))
                M.writeLengthBool(forbidOverlap);
            if (chunks.wasNext(0x24))
                M.writeLengthMultibyte(animationType);
            if (chunks.wasNext(0x25))
                M.writeLengthMultibyte(animationFrequency);
            
            if (chunks.wasNext(0x29))
                customRoute.write();
            
            if (chunks.wasNext(0x33))
                M.writeLengthMultibyte(getCommandsLength());
            
            if (chunks.wasNext(0x34))
                M.writeCommandList(commands);
            
            M.writeByte(0x00);
        }
        
        // Returns the byte length of just the commands section.
        public int getCommandsLength()
        {
            int length = 0;
            foreach (Command command in commands)
                length += command.getLength();
            return length;
        }
        
        // Sets sprite and animation settings, and ensures chunks for them are added.
        public void setAnimation(string charSet, int charIndex, int charDir, int charPattern, int animationType, int animationFrequency)
        {
            this.charSet = charSet;
            this.charIndex = charIndex;
            this.charDir = charDir;
            this.charPattern = charPattern;
            this.animationType = animationType;
            this.animationFrequency = animationFrequency;
            
            chunks.add(0x15); // charSet
            chunks.add(0x16); // charIndex
            chunks.add(0x17); // charDir
            chunks.add(0x18); // charPattern
            chunks.add(0x24); // animationType
            chunks.add(0x25); // animationFrequency
        }
        
        // Sets movement type/frequency/route, and ensures chunks for them are added.
        public void setMoveType(int moveType, int moveFrequency = 3, PageCustomRoute customRoute = null)
        {
            this.moveType = moveType;
            this.moveFrequency = moveFrequency;
            this.customRoute = customRoute;
            
            chunks.add(0x1f); // moveType
            chunks.add(0x20); // moveFrequency
            
            if (customRoute != null)
                chunks.add(0x29); // customRoute
            else
                chunks.remove(0x29); // customRoute
        }
        
        // Sets event layer and forbid overlap setting, and ensures chunks for them is added.
        public void setEventLayer(int layer, bool forbidOverlap = false)
        {
            this.layer = layer;
            this.forbidOverlap = forbidOverlap;
            
            chunks.add(0x22); // layer
            chunks.add(0x23); // allowOverlap
        }
    }
    
    class PageConditions : RPGByteData
    {
        bool switch1Condition = false; // 01 bits
        bool switch2Condition = false; // 01 bits
        bool variableCondition = false; // 01 bits
        bool itemCondition = false; // 01 bits
        bool heroCondition = false; // 01 bits
        bool timerCondition = false; // 01 bits
        bool timer2Condition = false; // 01 bits
        int switch1Num = 1; // 02
        int switch2Num = 1; // 03
        int variableNum = 1; // 04
        int variableValue = 0; // 05
        int itemValue = 1; // 06
        int heroNumber = 1; // 07
        int timerValue = 0; // 08
        int timer2Value = 0; // 09 (2003)
        int variableComparison = 1; // 0a (2003)
        
        static string myClass = "PageConditions";
        Chunks chunks;
        
        public PageConditions(FileStream f)
        {
            load(f);
        }
        public PageConditions()
        {
            chunks = new Chunks();
        }
        
        // Loads the conditions section.
        override public void load(FileStream f)
        {
            chunks = new Chunks(f, myClass);
            
            if (chunks.next(0x01))
            {
                bool[] flags = M.readLengthFlags(f);
                switch1Condition = flags[0];
                switch2Condition = flags[1];
                variableCondition = flags[2];
                itemCondition = flags[3];
                heroCondition = flags[4];
                timerCondition = flags[5];
                timer2Condition = flags[6];
            }
            if (chunks.next(0x02))
                switch1Num = M.readLengthMultibyte(f);
            if (chunks.next(0x03))
                switch2Num = M.readLengthMultibyte(f);
            if (chunks.next(0x04))
                variableNum = M.readLengthMultibyte(f);
            if (chunks.next(0x05))
                variableValue = M.readLengthMultibyte(f);
            if (chunks.next(0x06))
                itemValue = M.readLengthMultibyte(f);
            if (chunks.next(0x07))
                heroNumber = M.readLengthMultibyte(f);
            if (chunks.next(0x08))
                timerValue = M.readLengthMultibyte(f);
            if (chunks.next(0x09))
                timer2Value = M.readLengthMultibyte(f);
            if (chunks.next(0x0a))
                variableComparison = M.readLengthMultibyte(f);
            
            M.byteCheck(f, 0x00);
        }
        
        // Writes conditions section, to parent writer by default, and returns the byte size of that section.
        override protected void myWrite()
        {
            if (chunks.wasNext(0x01))
                M.writeLengthFlags(new bool[] { switch1Condition, switch2Condition, variableCondition,
                                                itemCondition, heroCondition, timerCondition, timer2Condition });
            if (chunks.wasNext(0x02))
                M.writeLengthMultibyte(switch1Num);
            if (chunks.wasNext(0x03))
                M.writeLengthMultibyte(switch2Num);
            if (chunks.wasNext(0x04))
                M.writeLengthMultibyte(variableNum);
            if (chunks.wasNext(0x05))
                M.writeLengthMultibyte(variableValue);
            if (chunks.wasNext(0x06))
                M.writeLengthMultibyte(itemValue);
            if (chunks.wasNext(0x07))
                M.writeLengthMultibyte(heroNumber);
            if (chunks.wasNext(0x08))
                M.writeLengthMultibyte(timerValue);
            if (chunks.wasNext(0x09))
                M.writeLengthMultibyte(timer2Value);
            if (chunks.wasNext(0x0a))
                M.writeLengthMultibyte(variableComparison);
            
            M.writeByte(0x00);
        }
        
        // Sets first switch condition, and ensures chunk for it is added.
        public void setSwitch1(int newSwitch)
        {
            switch1Condition = true;
            switch1Num = newSwitch;
            
            chunks.add(0x01); // bits
            chunks.add(0x02); // switch1Num
        }
        
        // Sets second switch condition, and ensures chunk for it is added.
        public void setSwitch2(int newSwitch)
        {
            switch2Condition = true;
            switch2Num = newSwitch;
            
            chunks.add(0x01); // bits
            chunks.add(0x03); // switch2Num
        }
    }
    
    class PageCustomRoute : RPGData
    {
        MoveRoute moveRoute; // 0b length, 0c content
        bool moveRepeat = true; // 15
        bool moveIgnore = false; // 16
        
        static string myClass = "PageCustomRoute";
        Chunks chunks;
        
        public PageCustomRoute(FileStream f)
        {
            load(f);
        }
        public PageCustomRoute(params MoveStep[] steps)
        {
            moveRoute = new MoveRoute(new List<MoveStep>(steps));
            
            chunks = new Chunks();
            chunks.add(0x0b); // length
            chunks.add(0x0c); // moveRoute
        }
        
        // Loads the custom route section.
        public void load(FileStream f)
        {
            chunks = new Chunks(f, myClass);
            
            M.readMultibyte(f); // Total move length
            
            if (chunks.next(0x0b))
                M.readLengthMultibyte(f);
            
            if (chunks.next(0x0c))
            {
                int moveLength = M.readMultibyte(f);
                if (moveLength > 0)
                    moveRoute = new MoveRoute(f, moveLength, "Custom");
            }
            
            if (chunks.next(0x15))
                moveRepeat = M.readLengthBool(f);
            if (chunks.next(0x16))
                moveIgnore = M.readLengthBool(f);
            
            M.byteCheck(f, 0x00);
        }
        
        // Writes custom move route section. Not the fancy way, though, because just getting the byte array length wouldn't work.
        public void write()
        {
            int moveLength = moveRoute != null? moveRoute.getLength() : 0;
            
            int totalMoveLength = 0;
            if (chunks.used(0x0b))
            {
                totalMoveLength += 1; // 0x0b
                totalMoveLength += M.countMultibyte(M.countMultibyte(moveLength)); // Length length
                totalMoveLength += M.countMultibyte(moveLength); // Move length
            }
            if (chunks.used(0x0c))
            {
                totalMoveLength += 1; // 0x0c
                totalMoveLength += M.countLengthMultibyte(moveLength); // Move length
                totalMoveLength += moveLength; // Actual route contents
            }
            if (chunks.used(0x15))
                totalMoveLength += 3; // 0x15, length (0x01), bool (0x00/0x01)
            if (chunks.used(0x16))
                totalMoveLength += 3; // 0x16, length (0x01), bool (0x00/0x01)
            
            M.writeMultibyte(totalMoveLength);
            
            if (chunks.wasNext(0x0b))
                M.writeLengthMultibyte(moveLength);
            
            if (chunks.wasNext(0x0c))
            {
                M.writeMultibyte(moveLength);
                if (moveLength > 0)
                    moveRoute.write();
            }
            
            if (chunks.wasNext(0x15))
                M.writeLengthBool(moveRepeat);
            if (chunks.wasNext(0x16))
                M.writeLengthBool(moveIgnore);
            
            M.writeByte(0x00);
        }
        
        // Adds a MoveStep.
        public void addStep(MoveStep step)
        {
            if (moveRoute == null)
                moveRoute = new MoveRoute();
            moveRoute.addStep(step);
        }
    }
}
